<?php

namespace App\Http\Controllers;


use PDF;
use App\User;
use App\Compra;
use App\CompraPago;
use App\Cliente;
use App\CompraEnvio;
use App\Producto;
use App\ProductoMedidas;
use App\Proveedor;

use App\ProveedorContactos;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ComprasController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        if (Auth::user()->id == 22) {
            return redirect()->route('compras.locales');
        }

        $compras = Compra::where('tipo_compra', 1)->orderBy('id', 'desc')
            ->with('proveedor')
            ->with('detalle')
            ->with('pagos')
            ->with('envio')
            ->get();
        return view('compras.dashboard', compact('compras'));
    }

    public function comprasLocales()
    {
        $compras = Compra::where('tipo_compra', 2)
            ->orderBy('created_at', 'desc')
            ->with('proveedor')
            ->with('detalle')
            ->with('pagos')
            ->with('envio')
            ->get();
        return view('compras.locales.dashboard', compact('compras'));
    }

    public function nuevaCompraLocal()
    {
        $proveedores = Proveedor::orderBy('nombre', 'asc')->get();
        $unidadesMedida = ProductoMedidas::all();
        return view('compras.locales.nueva', compact('proveedores', 'unidadesMedida'));
    }

    public function guardarCompraLocal(Request $request)
    {
        DB::beginTransaction();
        try {
            $correlativo = $this->generar_correlativo(2);

            // Crear colección directamente
            $productos = collect($request->id_p)->map(function ($valor, $indice) use ($request) {
                return [
                    'descripcion' => $valor === null ? $request->descripcion_p[$indice] : null,
                    'cantidad' => $request->cantidad_p[$indice],
                    'precio' => $request->precio_p[$indice],
                    'producto_id' => $valor,
                    'unidad_id' => $request->id_medida[$indice] ?? 1,
                ];
            });

            // Crear la compra
            $nuevaCompra = Compra::create([
                'correlativo' => $correlativo,
                'total_factura' => $productos->sum(fn($producto) => $producto['precio'] * $producto['cantidad']),
                'observaciones' => $request->observaciones,
                'condiciones' => $request->dias_credito ?? null,
                'condiciones_pago' => $request->condiciones_pago,
                'direccion' => $request->direccion,
                'tipo_cambio' => $request->tipo_cambio ?? 1,
                'tipo_compra' => 2,
                'proveedor_id' => $request->proveedor,
                'moneda_id' => $request->id_moneda,
            ]);

            // Crear el detalle de productos
            $nuevaCompra->detalle()->createMany($productos->toArray());

            DB::commit();

            return redirect()->route('compras.locales')->with([
                'message' => 'Compra guardada correctamente',
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();

            return $th->getMessage();

            // return redirect()->route('compras.locales')->withErrors([
            //     'message' => 'Error al guardar la compra',
            //     'error' => $th->getMessage(),
            // ]);
        }
    }

    public function descargarCompraLocal($compra_id)
    {
        $compra = Compra::where('id', $compra_id)
            ->with(['proveedor', 'detalle'])
            ->firstOrFail();

        // // Define el tamaño del PDF: Media carta (5.5 x 8.5 pulgadas)
        // $pdfConfig = [0, 0, 396, 612];

        // // Genera el PDF
        $pdf = Pdf::loadView('compras.locales.descargar-pdf', compact('compra'));

        // Retorna el PDF para su descarga
        return $pdf->stream('Orden_compra_' . $compra->correlativo . '.pdf');

        // return view('compras.locales.descargar-pdf', compact('compra'));
    }

    public function buscar_proveedor(Request $request)
    {
        $proveedor = Proveedor::find($request->id_proveedor);

        return response()->json($proveedor);
    }

    public function buscarProducto(Request $request)
    {
        $isCompraLocal = $request->tipo_compra == 2;
        $tipoProducto = $isCompraLocal ? 9 : 8;
        $query = strtoupper($request->term);
        $results = Producto::where("id_tipo", $tipoProducto)->where("deshabilitar", 0)->where("nombre", "LIKE", "%" . $query . "%")->with('medida')->get();

        $data = [];

        foreach ($results as $result) {
            $data[] = [
                'label' => $result->nombre,
                'id' => $result->id,
                'precio' => $result->precio_venta,
                'unidad' => $result->medida->nombre ?? '',
                'factor_conversion_kilos' => $result->factor_conversion_kilos,
                'micras' => $result->micras,
                'caracteristicas' => $result->tipos,
                'id_unidad' => $result?->medida?->id
            ];
        }

        return response()->json($data);
    }

    public function guardar_importacion(Request $request)
    {
        DB::beginTransaction();
        try {
            $archivo_proforma = $request->hasFile('proforma_file') ? $this->guardar_proforma_file($request) : null;
            $productos = [];
            foreach ($request->id_p as $indice => $valor) {
                $producto = [
                    'cantidad' => $request->cantidad_p[$indice],
                    'precio' => $request->precio_p[$indice],
                    'producto_id' => $valor,
                    'unidad_id' => $request->id_medida[$indice],
                ];
                array_push($productos, $producto);
            }
            // convertir productos a coleccion
            $productos = collect($productos);

            $nuevaCompra = new Compra();
            $nuevaCompra->correlativo = $this->generar_correlativo(1);
            //totak de la compra = precio * cantidad de todos los productos
            $nuevaCompra->total_pi = $productos->sum(function ($producto) {
                return $producto['precio'] * $producto['cantidad'];
            });
            $nuevaCompra->no_pi = $request->numero_pi;
            $nuevaCompra->observaciones = $request->observaciones;
            $nuevaCompra->condiciones = $request->condiciones_entrega;
            $nuevaCompra->condiciones_pago = $request->condiciones_pago;
            $nuevaCompra->incoterm = $request->incoterm;
            $nuevaCompra->tipo_cambio = $request->tipo_cambio ?? 1;
            $nuevaCompra->proforma_documento = $archivo_proforma;
            $nuevaCompra->proveedor_id = $request->proveedor;
            $nuevaCompra->moneda_id = $request->id_moneda;

            $nuevaCompra->save();

            foreach ($productos as $producto) {
                $nuevaCompra->detalle()->create([
                    'cantidad' => $producto['cantidad'],
                    'precio' => $producto['precio'],
                    'producto_id' => $producto['producto_id'],
                    'unidad_id' => $producto['unidad_id'],
                ]);
            }

            $arrResponse = [
                'message' => 'Compra guardada correctamente'
            ];
            DB::commit();
            return redirect()->route('compras.dashboard')->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $arrResponse = [
                'message' => 'Error al guardar la compra',
                'error' => $th->getMessage()
            ];
        }
    }

    public function actualizar_importacion(Request $request)
    {
        // dd($request->all());
        DB::beginTransaction();
        try {

            if ($request->revision == "on") {
                $compraUpdate = Compra::find($request->id_compra);
                $compraUpdate->estado = 3;
                $compraUpdate->save();
                $arrResponse = [
                    'message' => 'Compra actualizada correctamente'
                ];
                DB::commit();
                return redirect()->route('compras.detalle', $request->id_compra)->with($arrResponse);
            }


            $archivo_factura = $request->hasFile('factura_file') ? $this->guardar_documento_file($request, 'factura') : null;
            $archivo_impuesto = $request->hasFile('impuesto_file') ? $this->guardar_documento_file($request, 'impuesto') : null;
            $archivo_duca = $request->hasFile('duca_file') ? $this->guardar_documento_file($request, 'duca') : null;

            $compraUpdate = Compra::find($request->id_compra);
            $compraUpdate->total_factura = $request->total_factura;
            $compraUpdate->total_dai = $request->total_dai;
            $compraUpdate->total_iva = $request->total_iva;
            $compraUpdate->tipo_cambio = $request->tipo_cambio;
            $compraUpdate->total_flete_seguro = $request->total_flete_seguro;
            $compraUpdate->no_factura = $request->no_factura;
            $compraUpdate->no_duca = $request->no_duca;
            $compraUpdate->observaciones = $request->observaciones;
            $compraUpdate->factura_documento = $archivo_factura ?? $compraUpdate->factura_documento;
            $compraUpdate->impuesto_documento = $archivo_impuesto ?? $compraUpdate->impuesto_documento;
            $compraUpdate->duca_documento = $archivo_duca ?? $compraUpdate->duca_documento;

            if ($request->total_factura && $request->no_factura && $compraUpdate->estado == 1) {
                $compraUpdate->estado = 2;
            }

            $compraUpdate->save();

            $compraProductos = [];

            foreach ($request->producto_cantidad as $key => $value) {
                $compraProductos[] = [
                    'id' => $request->producto_id[$key],
                    'cantidad' => $value,
                    'precio' => $request->producto_precio[$key]
                ];
            }

            $compraProductos = collect($compraProductos);

            // Actualizar la cantidad de productos en la compra
            foreach ($compraUpdate->detalle as $compra) {
                $producto = $compraProductos->firstWhere('id', $compra->id);
                if ($producto) {
                    $compra->cantidad = $producto['cantidad'];
                    $compra->precio = $producto['precio'];
                    $compra->save();
                }
            }

            $arrResponse = [
                'message' => 'Compra actualizada correctamente'
            ];
            DB::commit();
            // Regresar a la ruta de detalle de la compra
            return redirect()->route('compras.detalle', $request->id_compra)->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $arrResponse = [
                'message' => 'Error al guardar cambios en la compra',
                'error' => $th->getMessage()
            ];
        }
    }

    public function actualizar_eta(Request $request)
    {

        // dd($request->all());
        DB::beginTransaction();

        try {
            $compraUpdate = Compra::findOrfail($request->compra_id);
            $compraUpdate->envio()->create([
                'detalles' => $request->detalles,
                'fecha_estimada' => $request->eta,
                'empresa' => $request->naviera,
                'guia' => $request->bl,
                'origen' => $request->origen,
                'destino' => $request->destino,
                'created_at' => now()
            ]);

            $arrResponse = [
                'message' => 'ETA actualizado correctamente'
            ];

            DB::commit();

            return redirect()->route('compras.detalle', $request->compra_id)->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $arrResponse = [
                'message' => 'Error al actualizar ETA en la compra',
                'error' => $th->getMessage()
            ];
        }
    }

    public function registrar_pago(Request $request)
    {
        DB::beginTransaction();
        try {
            $archivos_pago = [];
            if ($request->hasFile('pago_file')) {
                foreach ($request->file('pago_file') as $archivo) {
                    $archivos_pago[] = $this->guardar_pago_file($archivo); // Asume que este método maneja cada archivo.
                }
            }

            $compraUpdate = Compra::find($request->id_compra);
            if ($request->tipo_pago == 1) {
                $compraUpdate->total_pagado += $request->monto;
                $compraUpdate->save();
            }

            $iva = ($request->iva == "on") ? 1 : 0;

            // Combina los archivos nuevos con los existentes, si aplica.
            $archivos_guardados = implode(',', $archivos_pago);

            $compraUpdate->pagos()->create([
                'monto' => $request->monto,
                'tipo_cambio' => $request->tipo_cambio,
                'no_documento' => $request->no_documento,
                'comprobante' => $archivos_guardados, // Guarda los nombres separados por comas.
                'detalles' => $request->detalles,
                'tipo_pago' => $request->tipo_pago,
                'moneda_id' => $request->moneda_id,
                'iva' => $iva,
                'user_id' => Auth::id(),
                'created_at' => now()
            ]);

            $arrResponse = [
                'message' => 'Pago registrado correctamente'
            ];
            DB::commit();

            return redirect()->route('compras.detalle', $request->id_compra)->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            dd($th->getMessage());
        }
    }


    public function editar_pago(Request $request)
    {
        DB::beginTransaction();
        try {
            $compraUpdate = Compra::find($request->id_compra);
            $pago = $compraUpdate->pagos()->find($request->id_pago);

            // Actualizar el monto total pagado según el tipo de pago
            if ($pago->tipo_pago == 1) {
                $compraUpdate->total_pagado -= $pago->monto;
            }
            if ($request->tipo_pago == 1) {
                $compraUpdate->total_pagado += $request->monto;
            }
            $compraUpdate->save();

            $iva = ($request->iva == "on") ? 1 : 0;

            $archivos_pago = [];
            if ($request->hasFile('pago_file')) {
                foreach ($request->file('pago_file') as $archivo) {
                    $archivos_pago[] = $this->guardar_pago_file($archivo);
                }
            }

            // Combinar archivos nuevos con los existentes
            $archivos_existentes = $pago->comprobante ? explode(',', $pago->comprobante) : [];
            $archivos_guardados = array_merge($archivos_existentes, $archivos_pago);

            $pago->update([
                'monto' => $request->monto,
                'moneda_id' => $request->moneda_id,
                'tipo_pago' => $request->tipo_pago,
                'tipo_cambio' => $request->tipo_cambio,
                'no_documento' => $request->no_documento,
                'detalles' => $request->detalles,
                'iva' => $iva,
                'comprobante' => implode(',', $archivos_guardados), // Guardar como lista separada por comas.
            ]);

            $arrResponse = [
                'message' => 'Pago actualizado correctamente'
            ];
            DB::commit();

            return redirect()->route('compras.detalle', $request->id_compra)->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with([
                'message' => 'Error al actualizar pago en la compra',
                'error' => $th->getMessage()
            ]);
        }
    }

    public function eliminar_pago(Request $request)
    {
        DB::beginTransaction();
        try {
            $pago = CompraPago::findOrFail($request->id);
            $compra = Compra::find($pago->compra_id);
            if ($pago->tipo_pago == 1) {
                $compra->total_pagado -= $pago->monto;
                $compra->save();
            }
            $pago->delete();
            DB::commit();
            return redirect()->route('compras.detalle', $compra->id)->with([
                'message' => 'Pago eliminado correctamente'
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with([
                'message' => 'Error al eliminar pago',
                'error' => $th->getMessage()
            ]);
        }
    }

    public function eliminarArchivos($id)
    {
        DB::beginTransaction();
        try {
            $pago = CompraPago::findOrFail($id);

            // Obtener los archivos actuales
            $archivos = $pago->comprobante ? explode(',', $pago->comprobante) : [];

            // Eliminar cada archivo del almacenamiento
            foreach ($archivos as $archivo) {
                if (Storage::exists("compras/pagos/{$archivo}")) {
                    Storage::delete("compras/pagos/{$archivo}");
                }
            }

            // Limpiar el campo de la base de datos
            $pago->comprobante = null;
            $pago->save();

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Archivos eliminados correctamente.']);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al eliminar archivos.', 'error' => $th->getMessage()], 500);
        }
    }


    public function generar_correlativo($tipo_compra = 1)
    {
        // Obtén el año actual
        $year = Carbon::now()->year;

        // Encuentra el último número de serie para el año actual y tipo de compra
        $ultimoNumero = Compra::where('correlativo', 'LIKE', "%-$year")
            ->where('tipo_compra', $tipo_compra)
            ->orderByDesc('id') // Garantiza orden por el último registro
            ->value('correlativo');

        // Generar el nuevo número
        $nuevoNumero = $ultimoNumero
            ? intval(explode('-', $ultimoNumero)[0]) + 1
            : 1;

        // Crear el nuevo correlativo en formato "01-Año"
        $nuevaSerie = sprintf('%02d-%d', $nuevoNumero, $year);

        return $nuevaSerie;
    }


    public function guardar_proforma_file(Request $request)
    {
        try {
            $file = $request->file('proforma_file');
            $nombre = $file->getClientOriginalName();
            $nombre = time() . '_' . $nombre;
            //Almacenar archivo en la ruta: storage/compras/proform_invoice/
            $file->move('storage/compras/proform_invoice/', $nombre);

            return $nombre;
        } catch (\Throwable $th) {
            return null;
        }
    }

    public function guardar_documento_file(Request $request, $nombre_file)
    {
        try {
            $file = $request->file($nombre_file . '_file');
            $nombre = $file->getClientOriginalName();
            $nombre = time() . '_' . $nombre;
            //Almacenar archivo en la ruta: storage/compras/proform_invoice/
            $file->move('storage/compras/' . $nombre_file . 's/', $nombre);

            return $nombre;
        } catch (\Throwable $th) {
            return null;
        }
    }

    public function guardar_pago_file($file)
    {
        try {
            $nombre = $file->getClientOriginalName();
            $nombre = time() . '_' . $nombre;
            //Almacenar archivo en la ruta: storage/compras/proform_invoice/
            $file->move('storage/compras/pagos/', $nombre);

            return $nombre;
        } catch (\Throwable $th) {
            return null;
        }
    }

    public function detalle($compra_id)
    {
        $compra = Compra::where('id', $compra_id)
            ->with('proveedor')
            ->with('detalle')
            ->with('pagos')
            ->with('envio')
            ->first();

        $compra_eta = $compra->envio->first() ?? new CompraEnvio();
        if ($compra->tipo_cambio === null || $compra->tipo_cambio < 1) {
            $compra->tipo_cambio = 1;
        }
        $compra->incoterm = match ($compra->incoterm) {
            1 => 'EXW',
            2 => 'FOB',
            3 => 'CIF',
            default => 'DESCONOCIDO'
        };
        $tipo_cambio = $compra->tipo_cambio;
        $total_factura = $compra->total_factura; //Dolares
        $total_dai = $compra->total_dai ?? 0; //Dolares
        $total_flete_seguro = $compra->total_flete_seguro; //Dolares

        $prorrateo_datos = null;
        if (count($compra->pagos) > 0) {
            $prorrateo_datos = new \stdClass();
            // Paso 1
            //Datos a obtener: pago en dolares a naviera, total de gastos locales sin iva y con iva, total de importacion (valor factura+valor flete+DAI+Pago en dolares naviera + total gastos locales, todo en quetzales).

            //Total de factura + flete y seguro
            $prorrateo_datos->valor_cif_factura = ($total_factura + $total_flete_seguro);
            $prorrateo_datos->valor_cif_factura_quetzales = ($prorrateo_datos->valor_cif_factura * $tipo_cambio) == 0 ? 1 : ($prorrateo_datos->valor_cif_factura * $tipo_cambio);

            //Total de dai en dolares y quetzales (el valor se guarda en dolares)
            $prorrateo_datos->total_dai = $total_dai ?? 0;
            $prorrateo_datos->total_dai_quetzales = $total_dai;
            //Total de pago naviera (tipo de pago entre 2 y 6)
            $prorrateo_datos->pago_naviera_dolares = $compra->pagos->where('tipo_pago', 2)->sum('monto') ?? 0;
            $prorrateo_datos->pago_naviera_dolares += $compra->pagos->where('tipo_pago', 3)->sum('monto') ?? 0;
            $prorrateo_datos->pago_naviera_quetzales = $prorrateo_datos->pago_naviera_dolares * $tipo_cambio;

            //Total de gastos locales
            $gastosLocales = $compra->pagos->where('tipo_pago', '>', 3);
            $gastosLocalesSinIvaQuetzales = $gastosLocales->where('iva', 0)->where('moneda_id', 1)->sum('monto');
            $gastosLocalesSinIvaDolaresAQuetzales = $gastosLocales->where('iva', 0)->where('moneda_id', 2)->map(function ($pago) use ($tipo_cambio) {
                return $pago->monto * $tipo_cambio;
            })->sum();

            // Gastos Locales con IVA
            $gastosLocalesConIvaQuetzales = $gastosLocales->where('iva', 1)->where('moneda_id', 1)->sum('monto') / 1.12 ?? 0;
            $gastosLocalesConIvaDolaresAQuetzales = $gastosLocales->where('iva', 1)->where('moneda_id', 2)->map(function ($pago) use ($tipo_cambio) {
                return ($pago->monto * $tipo_cambio) / 1.12;
            })->sum() ?? 0;

            $prorrateo_datos->gastos_locales_sin_iva = $gastosLocalesSinIvaQuetzales + $gastosLocalesSinIvaDolaresAQuetzales + $gastosLocalesConIvaQuetzales + $gastosLocalesConIvaDolaresAQuetzales;
            $prorrateo_datos->gastos_locales_sin_iva_dolares = $prorrateo_datos->gastos_locales_sin_iva / $tipo_cambio;

            // var_dump($prorrateo_datos);
            // Paso 2
            $prorrateo_datos->total_importacion_quetzales = $prorrateo_datos->valor_cif_factura_quetzales + $prorrateo_datos->total_dai_quetzales + $prorrateo_datos->pago_naviera_quetzales + $prorrateo_datos->gastos_locales_sin_iva;
            // $prorrateo_datos->total_importacion_dolares = $prorrateo_datos->valor_cif_factura + $prorrateo_datos->total_dai + $prorrateo_datos->pago_naviera_dolares + $prorrateo_datos->gastos_locales_sin_iva_dolares;

            //Paso 3 coeficientes
            $prorrateo_datos->coeficiente1 = number_format($prorrateo_datos->pago_naviera_quetzales / $prorrateo_datos->valor_cif_factura_quetzales, 6);
            $prorrateo_datos->coeficiente2 = number_format($prorrateo_datos->gastos_locales_sin_iva / $prorrateo_datos->valor_cif_factura_quetzales, 6);
        }

        return view('compras.detalle', compact('compra', 'prorrateo_datos', 'compra_eta'));
    }

    public function ordenCompra()
    {
        //
        return view('compras.orden_compra');
    }

    public function nueva_importacion()
    {
        //Fecha +15 dias
        $date_now = date('d-m-Y');
        $date_future = strtotime('+15 day', strtotime($date_now));
        $date_future = date('d-m-Y', $date_future);
        //
        $proveedores = Proveedor::orderBy('nombre', 'asc')->get();
        return view('compras.nueva_importacion', compact('proveedores', 'date_future'));
    }

    public function dashboardProveedor()
    {
        //
        $clientes = Proveedor::orderBy('nombre', 'asc')->get();
        return view('compras.dashboard_proveedor', compact('clientes'));
    }
    public function nuevoProveedor()
    {
        //
        $ejecutivos = User::where('role', 2)->get();
        return view('compras.proveedor', compact('ejecutivos'));
    }
    public function editarProveedor($idProveedor)
    {
        //
        $cliente = Proveedor::where('id', $idProveedor)->with('contactos')->first();
        return view('compras.editar_proveedor', compact('cliente'));
    }
    public function actualizarProveedor(Request $request, $idProveedor)
    {
        $cliente = Proveedor::where('id', $idProveedor)->first();
        //
        $cliente->id_usuario =  Auth::id();
        $cliente->nombre = $request->nombre;
        $cliente->nit = $request->nit;
        $cliente->direccion = $request->direccion;
        $cliente->correo_electronico = $request->correo_electronico;
        $cliente->sitio_web = $request->sitio_web;
        $cliente->puesto = $request->puesto;
        $cliente->nombre_contacto = $request->nombre_contacto1;
        $cliente->telefono = $request->telefono;
        $cliente->celular = $request->celular;
        $cliente->condiciones_pago = $request->condiciones_pago;
        $cliente->ejecutivo = 0;
        $cliente->referencia = $request->referencia;
        $cliente->p_interior = $request->p_interno ?? 0;
        $cliente->p_exterior = $request->p_externo ?? 0;
        $cliente->pais = $request->pais ?? '';
        $cliente->update();

        ProveedorContactos::where('id_cliente', $idProveedor)->delete();
        for ($i = 0; $i <= 1; $i++) {
            if ($request->nombre_contacto[$i]) {
                $nuevoContacto = new ProveedorContactos();
                $nuevoContacto->id_cliente = $idProveedor;
                $nuevoContacto->nombre_contacto = $request->nombre_contacto[$i];
                $nuevoContacto->celular = $request->celular_contacto[$i];
                $nuevoContacto->correo_electronico = $request->correo_electronico_contacto[$i];
                $nuevoContacto->puesto = $request->puesto__contacto[$i];
                $nuevoContacto->save();
            }
        }
        $arrResponse = [
            'message' => 'Proveedor guardado correctamente'
        ];
        return redirect()->route('compras.dashboard_proveedor')->with($arrResponse);
    }
    public function eliminarProveedor($id)
    {
        //
        Proveedor::destroy($id);
        ProveedorContactos::where('id_cliente', $id)->delete();
        $arrResponse = [
            'message' => 'Proveedor eliminado correctamente'
        ];
        return redirect()->route('compras.dashboard_proveedor')->with($arrResponse);
    }
    public function guardar_proveedor(Request $request)
    {
        //
        $cliente = new Proveedor();
        $cliente->id_usuario =  Auth::id();
        $cliente->nombre = $request->nombre;
        $cliente->nit = $request->nit;
        $cliente->direccion = $request->direccion;
        $cliente->correo_electronico = $request->correo_electronico;
        $cliente->sitio_web = $request->sitio_web;
        $cliente->puesto = $request->puesto;
        $cliente->nombre_contacto = $request->nombre_contacto1;
        $cliente->telefono = $request->telefono;
        $cliente->celular = $request->celular;
        $cliente->ejecutivo = 0;
        $cliente->condiciones_pago = $request->condiciones_pago;
        $cliente->referencia = $request->referencia;

        $cliente->p_interior = $request->p_interno ?? 0;
        $cliente->p_exterior = $request->p_externo ?? 0;
        $cliente->pais = $request->pais ?? '';
        $cliente->save();
        for ($i = 0; $i <= 1; $i++) {
            if ($request->nombre_contacto[$i]) {
                $nuevoContacto = new ProveedorContactos();
                $nuevoContacto->id_cliente = $cliente->id;
                $nuevoContacto->nombre_contacto = $request->nombre_contacto[$i];
                $nuevoContacto->celular = $request->celular_contacto[$i];
                $nuevoContacto->correo_electronico = $request->correo_electronico_contacto[$i];
                $nuevoContacto->puesto = $request->puesto__contacto[$i];
                $nuevoContacto->save();
            }
        }
        $arrResponse = [
            'message' => 'Proveedor actualizado correctamente'
        ];
        return redirect()->route('compras.dashboard_proveedor')->with($arrResponse);
    }

    // Compras locales
    public function uploadCotizacion(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $request->validate([
                'proforma_documento' => 'required|mimes:pdf|max:5128', // Solo PDFs de hasta 2MB
            ]);

            $compra = Compra::findOrFail($id);

            // Eliminar el archivo anterior si existe
            if ($compra->proforma_documento) {
                Storage::delete('public/compras/locales/' . $compra->proforma_documento);
            }

            // Guardar el nuevo archivo
            $file = $request->file('proforma_documento');
            $filename = 'cotizacion_' . $compra->id . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public/compras/locales', $filename);

            // Guardar el nombre del archivo en la base de datos
            $compra->proforma_documento = $filename;
            $compra->save();

            $arrResponse = [
                'message' => 'Cotización cargada con éxito.'
            ];

            DB::commit();

            return redirect()->back()->with($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->withErrors([
                'error' => 'Error al cargar la cotización: ' . $th->getMessage(),
            ]);
        }
    }


    // HELPERS
    function safeDivide($dividendo, $divisor): float
    {
        return $dividendo / ($divisor != 0 ? $divisor : 1);
    }
}
