<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Retencion extends Model
{
    use SoftDeletes;

    protected $table = 'retenciones';

    protected $fillable = [
        'numero',
        'monto_total',
        'observaciones',
        'tipo',
        'estado',
        'moneda_id',
        'user_id'
    ];

    // Relacion muchos a muchos con factura y retencion a traves del modelo pivote RetencionFactura

    public function facturas()
    {
        return $this->belongsToMany('App\Facturas', 'retencion_facturas', 'retencion_id', 'factura_id')->withPivot('monto');
    }

    public function user()
    {
        return $this->belongsTo('App\User');
    }

    // Funcion auxiliar para obtener el tipo de retencion como un enum: 1 = IVA, 2 = ISR

    public function getTipoStringAttribute(): string
    {
        return match ($this->tipo) {
            1 => 'IVA',
            2 => 'ISR',
            default => 'Desconocido'
        };
    }

    // Funcion auxiliar para obtener el estado de la retencion como un enum: 1 = Activa, 2 = Revertida, 3 = Anulada

    public function getEstadoStringAttribute(): string
    {
        return match ($this->estado) {
            1 => 'Activo',
            2 => 'Revertida',
            3 => 'Anulada',
            default => 'Desconocido'
        };
    }

    public function getEstadoColorAttribute(): string
    {
        return match ($this->estado) {
            1 => '',
            2 => 'bg-warning',
            3 => 'bg-danger',
            default => 'Desconocido'
        };
    }

    // Obtener Cliente por medio de la relacion con la primera factura asociada a la retencion
    // Accessor para obtener el cliente de la primera factura asociada a la retención
    public function getClienteAttribute()
    {
        // Cargar la primera factura con la relación cliente
        $factura = $this->facturas()->with('cliente')->first();

        // Retornar el cliente de la factura si existe
        return $factura?->cliente;
    }
}
